<?php
session_start();
require_once 'inc/dbconn.php';

$userid = $_SESSION['uid'] ?? null;
if (!$userid) {
    die("Not logged in");
}

$directory = __DIR__ . '/uploads/' . $userid; // absolute path is safer
if (!is_dir($directory)) {
    mkdir($directory, 0770, true);
}

if (!empty($_FILES)) {
    $file = $_FILES['file'];
    if ($file['error'] !== UPLOAD_ERR_OK) {
        die("Upload failed with error code " . $file['error']);
    }

    // validate file type
    $allowedMimes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/jfif'];
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    if (!in_array($mime, $allowedMimes, true)) {
        die("Invalid file type: $mime");
    }

    // load source image
    switch ($mime) {
        case 'image/jpeg':
            $src = imagecreatefromjpeg($file['tmp_name']);
            break;
        case 'image/png':
            $src = imagecreatefrompng($file['tmp_name']);
            break;
        case 'image/webp':
            $src = imagecreatefromwebp($file['tmp_name']);
            break;
        default:
            die("Unsupported image type");
    }

    if (!$src) {
        die("Failed to decode image");
    }

    // resize/compress
    $maxW = 1200; // adjust as needed
    $maxH = 1200;
    $quality = 80; // compression quality

    [$width, $height] = getimagesize($file['tmp_name']);
    $scale = min(1.0, $maxW / $width, $maxH / $height);

    $dst = $src;
    if ($scale < 1.0) {
        $newW = (int)floor($width * $scale);
        $newH = (int)floor($height * $scale);
        $dst = imagecreatetruecolor($newW, $newH);
        imagealphablending($dst, false);
        imagesavealpha($dst, true);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $newW, $newH, $width, $height);
        imagedestroy($src);
    }

    // generate safe filename
    $ext = match($mime) {
        'image/jpeg' => 'jpg',
        'image/png'  => 'png',
        'image/webp' => 'webp',
    };
    $finalName = bin2hex(random_bytes(12)) . '.' . $ext;
    $finalPath = $directory . '/' . $finalName;

    $ok = false;
    switch ($ext) {
        case 'jpg':
            $ok = imagejpeg($dst, $finalPath, $quality);
            break;
        case 'png':
            $pngQ = (int)round((100 - $quality) * 9 / 100);
            $ok = imagepng($dst, $finalPath, $pngQ);
            break;
        case 'webp':
            $ok = imagewebp($dst, $finalPath, $quality);
            break;
    }

    imagedestroy($dst);

    if (!$ok) {
        die("Failed to save compressed image");
    }

    chmod($finalPath, 0640);

    echo "Image uploaded and compressed successfully: $finalName";
}
