<?php
// new_product_handler.php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

session_start();
date_default_timezone_set('Africa/Nairobi');

include_once "inc/dbconn.php"; // must set $connection (mysqli)

if (!isset($connection) || !$connection instanceof mysqli) {
    die("<script>alert('Database connection not found');location.href='new_product.php';</script>");
}

// Make mysqli throw exceptions so we can rollback cleanly
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Helper: fail and redirect
function fail($msg) {
    echo "<script>alert(" . json_encode($msg) . ");location.href='new_product.php';</script>";
    exit();
}

// Input retrieval & basic validation (use null coalescing to avoid undefined index warnings)
$userid = isset($_SESSION['uid']) ? intval($_SESSION['uid']) : null;
$product_name = isset($_POST['product_name']) ? trim($_POST['product_name']) : '';
$brand_id = isset($_POST['brand_id']) ? intval($_POST['brand_id']) : null;
$category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : null;
$product_desc = isset($_POST['product_desc']) ? trim($_POST['product_desc']) : '';
$product_qty = isset($_POST['product_qty']) ? intval($_POST['product_qty']) : 0;
$product_price = isset($_POST['product_price']) ? $_POST['product_price'] : '0';
$product_short_description = isset($_POST['product_short_description']) ? trim($_POST['product_short_description']) : '';
$product_keywords = isset($_POST['meta_keywords']) ? trim($_POST['meta_keywords']) : '';
$product_condition = isset($_POST['product_condition']) ? trim($_POST['product_condition']) : '';
$product_status = isset($_POST['product_status']) ? trim($_POST['product_status']) : '';
$meta_title = isset($_POST['meta_title']) ? trim($_POST['meta_title']) : '';
$meta_description = isset($_POST['meta_description']) ? trim($_POST['meta_description']) : '';

if (empty($userid)) {
    fail('User not authenticated. Please login and try again.');
}

if ($product_name === '') {
    fail('Title should not be empty');
}

// sanitize product price
if (!is_numeric($product_price)) {
    fail('Product price must be a valid number.');
}
$product_price = floatval($product_price);

// create slug safely (lowercase, ascii-compatible)
function make_slug($text) {
    $text = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $text); // transliterate
    $text = preg_replace('/[^A-Za-z0-9-]+/', '-', $text);
    $text = preg_replace('/-+/', '-', $text);
    $text = trim($text, '-');
    return strtolower($text);
}

$base_slug = make_slug($product_name);
$slug = $base_slug;

// Ensure unique slug: attempt select, if exists append -n
try {
    $suffix = 0;
    $stmt_check = $connection->prepare("SELECT COUNT(*) FROM products WHERE slug = ?");
    while (true) {
        $stmt_check->bind_param('s', $slug);
        $stmt_check->execute();
        $stmt_check->bind_result($cnt);
        $stmt_check->fetch();
        $stmt_check->reset();
        if ($cnt == 0) break;
        $suffix++;
        $slug = $base_slug . '-' . $suffix;
    }
    $stmt_check->close();
} catch (Exception $e) {
    fail('Database error while checking slug: ' . $e->getMessage());
}

// Validate description
if ($product_desc === '') {
    fail('Product description should not be empty');
}

// Prepare description and meta files
$ff_title = mb_substr($product_name, 0, 6);
$desc_filename_base = $ff_title . "@" . uniqid("smartphonecityke", true);
$desc_filename = $desc_filename_base . ".txt";
$desc_path = __DIR__ . "/../description/" . $desc_filename;

$meta_filename_base = $ff_title . "@" . uniqid("_smartphonecityke", true);
$meta_filename = $meta_filename_base . ".txt";
$meta_path = __DIR__ . "/../metadesc/" . $meta_filename;

// Normalize newlines -> use <br> for storing HTML friendly text
$desc_content = preg_replace('/\v+|\r\n|\n|\r/u','<br>', $product_desc);
$meta_content = preg_replace('/\v+|\r\n|\n|\r/u','<br>', $meta_description);

// Create directories if not exist (safe)
if (!is_dir(__DIR__ . "/../description/")) {
    @mkdir(__DIR__ . "/../description/", 0755, true);
}
if (!is_dir(__DIR__ . "/../metadesc/")) {
    @mkdir(__DIR__ . "/../metadesc/", 0755, true);
}
if (!is_dir(__DIR__ . "/../product_images/")) {
    @mkdir(__DIR__ . "/../product_images/", 0755, true);
}

$desc_written = @file_put_contents($desc_path, $desc_content);
$meta_written = @file_put_contents($meta_path, $meta_content);

if ($desc_written === false || $meta_written === false) {
    // cleanup any partial file
    if (file_exists($desc_path)) @unlink($desc_path);
    if (file_exists($meta_path)) @unlink($meta_path);
    fail('Failed to write description or meta files on server.');
}

// Image processing: gather user's uploaded files
$insertValuesSQL = ""; // list of filenames to store in DB, comma-separated
$errorUpload = "";
$errorUploadType = "";
$allowed_exts = array('gif','jpg','jpeg','png','jfif','webp');

$uploadPattern = __DIR__ . "/uploads/" . $userid . "/*.*";
$all_files = glob($uploadPattern);

$targetDir = __DIR__ . "/../product_images/";

$copied_files = []; // keep track to remove on rollback if needed

foreach ($all_files as $image_path) {
    // skip if not a file
    if (!is_file($image_path)) continue;

    $ext = strtolower(pathinfo($image_path, PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed_exts)) {
        $errorUploadType .= ($errorUploadType ? ' | ' : '') . basename($image_path);
        continue;
    }

    $imageRealName = basename($image_path);
    $imageFileType = $ext;

    $ff = mb_substr($product_name, 0, 6);
    $safeNameFragment = preg_replace('/[^A-Za-z0-9_-]/', '', $ff);
    $fileName = $safeNameFragment . "_crownbeauty_" . uniqid() . "." . $imageFileType;
    $targetFilePath = $targetDir . $fileName;

    // Copy file (safer than rename) then later unlink original if everything succeeds
    if (@copy($image_path, $targetFilePath)) {
        // set permissions
        @chmod($targetFilePath, 0644);
        $insertValuesSQL .= $fileName . ",";
        $copied_files[] = ['src' => $image_path, 'dest' => $targetFilePath];
    } else {
        $errorUpload .= ($errorUpload ? ' | ' : '') . basename($image_path);
    }
}

// Trim trailing comma
$insertValuesSQL = rtrim($insertValuesSQL, ',');

// If no images were copied, decide whether that's fatal or acceptable.
// Here we treat it as acceptable but warn.
if (empty($insertValuesSQL)) {
    $errorMsg = "No valid product images were uploaded.";
} else {
    $errorMsg = "";
}

// Database insert inside transaction
$time = time();

try {
    $connection->begin_transaction();

    $sql = "INSERT INTO products 
        (product_cat, product_brand, product_title, slug, product_qty, product_price, product_desc, product_image, product_keywords, product_short_description, product_condition, product_status, meta_title, meta_description, time)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $stmt = $connection->prepare($sql);

    // types: category (i), brand (i), product_name (s), slug (s), qty (i), price (d), desc filename (s),
    // product_image (s), product_keywords (s), short_description (s), condition (s), status (s), meta_title (s), meta_filename (s), time (i)
    $types = "iissidssssssssi";
    $stmt->bind_param(
        $types,
        $category_id,
        $brand_id,
        $product_name,
        $slug,
        $product_qty,
        $product_price,
        $desc_filename,
        $insertValuesSQL,
        $product_keywords,
        $product_short_description,
        $product_condition,
        $product_status,
        $meta_title,
        $meta_filename,
        $time
    );

    $stmt->execute();
    $stmt->close();

    // On success, remove original uploaded files (we copied earlier)
    foreach ($copied_files as $f) {
        if (file_exists($f['src'])) {
            @unlink($f['src']); // best-effort cleanup
        }
    }

    $connection->commit();

    // Success
    $msg = 'Product added successfully.';
    if (!empty($errorMsg) || !empty($errorUpload) || !empty($errorUploadType)) {
        $msg .= " Note: " . trim($errorMsg . ' ' . ($errorUpload ? "Upload errors: $errorUpload." : '') . ' ' . ($errorUploadType ? "Invalid types: $errorUploadType." : ''));
    }

    echo "<script>alert(" . json_encode($msg) . ");location.href='new_product.php';</script>";
    exit();

} catch (Exception $e) {
    // Rollback DB, remove any copied files and restore originals if necessary
    $connection->rollback();

    // remove any copied dest files
    foreach ($copied_files as $f) {
        if (file_exists($f['dest'])) @unlink($f['dest']);
    }

    // remove created desc/meta files
    if (file_exists($desc_path)) @unlink($desc_path);
    if (file_exists($meta_path)) @unlink($meta_path);

    fail('Failed to add product: ' . $e->getMessage());
}
